<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * class-ss-free-features.php
 *
 * Free Features Page — AI Checks (Free)
 * Clean, modern UI for ALL plans
 *
 * Save to:
 * /public_html/wp-content/plugins/searchshifter/includes/class-ss-free-features.php
 */

class SS_Free_Features {

    const TRANSIENT_PREFIX = 'ss_free_checks_'; // prefix for per-post transient
    const TRANSIENT_TTL    = 10 * MINUTE_IN_SECONDS; // 10 minutes cache

    public static function init() {
        add_action('admin_menu', [__CLASS__, 'register_menu'], 12);
        add_action('admin_head', [__CLASS__, 'admin_styles']);
        add_action('admin_footer', [__CLASS__, 'admin_scripts']);
    }

    /**
     * Add custom CSS for beautiful UI
     */
    public static function admin_styles() {
        ?>
        <style>
            .ss-section {
                background: #fff;
                border: 1px solid #e1e4e8;
                border-radius: 8px;
                padding: 20px;
                margin-top: 25px;
                box-shadow: 0 1px 0 rgba(10,10,10,0.02);
            }
            .ss-section h2 {
                margin: 0 0 12px 0 !important;
                padding-bottom: 6px;
                border-bottom: 1px solid #f1f1f1;
                font-size: 18px;
            }
            .ss-badge-green { color: #1e7e34; font-weight: 600; }
            .ss-badge-red   { color: #c92a2a; font-weight: 600; }
            .ss-cta {
                background: #f2fbff;
                border-left: 4px solid #2271b1;
                padding: 14px 16px;
                border-radius: 4px;
                margin-top: 18px;
            }
            .ss-table td, .ss-table th { vertical-align: middle !important; }
            .ss-filter-bar { display:flex; gap:12px; align-items:center; margin-bottom:12px; }
            .ss-mini { font-size:13px;color:#666; }
            .ss-issue-count {
                background:#fff; border:1px solid #e6e6e6; padding:6px 10px; border-radius:6px;
            }
            .ss-empty { color:#666; font-style:italic; }
            .ss-small-link { font-size:13px; color:#0a58ca; text-decoration:underline; margin-right:6px; }
            .ss-button-edit { font-size:13px; padding:4px 8px; }
            /* responsive table in admin */
            @media (max-width:800px) {
                .ss-table thead { display:none; }
                .ss-table tr { display:block; margin-bottom:10px; border-bottom:1px solid #eee; padding-bottom:10px; }
                .ss-table td { display:block; width:100%; }
            }
        </style>
        <?php
    }

    /**
     * Small JS for toggling 'only affected' view
     */
    public static function admin_scripts() {
        ?>
        <script>
        (function(){
            document.addEventListener('DOMContentLoaded', function(){
                var btn = document.getElementById('ss-toggle-affected');
                if (!btn) return;
                btn.addEventListener('click', function(e){
                    e.preventDefault();
                    var table = document.getElementById('ss-issues-table');
                    if (!table) return;
                    var rows = table.querySelectorAll('tbody tr');
                    var showingOnly = btn.getAttribute('data-only') === '1';
                    if (!showingOnly) {
                        rows.forEach(function(r){
                            // second cell contains page list; if it contains "All good" hide it
                            var td = r.querySelector('td:nth-child(2)');
                            if (td && td.textContent.trim().indexOf('All good') !== -1) {
                                r.style.display = 'none';
                            } else {
                                r.style.display = '';
                            }
                        });
                        btn.textContent = 'Show all issues';
                        btn.setAttribute('data-only','1');
                    } else {
                        rows.forEach(function(r){ r.style.display = ''; });
                        btn.textContent = 'Show only affected pages';
                        btn.setAttribute('data-only','0');
                    }
                });
            });
        })();
        </script>
        <?php
    }

    /**
     * Register menu
     */
public static function register_menu() {

    // Get plan from licensing system
    if (class_exists('SS_Licensing')) {
        $plan = strtolower(SS_Licensing::plan());

        // Hide this menu on Pro/Authority/Elite
        if (in_array($plan, ['pro', 'authority', 'elite'])) {
            return;
        }
    }

    // Show ONLY on free plan
    add_submenu_page(
        'searchshifter',
        __('AI Checks (Free)', 'searchshifter'),
        __('AI Checks (Free)', 'searchshifter'),
        'manage_options',
        'searchshifter-free-checks',
        [__CLASS__, 'render_page']
    );
}


    /**
     * Render Page
     */
    public static function render_page() {

        // gather some basic things
        $robots_url = site_url('/robots.txt');
        $ai_txt     = ABSPATH . 'ai.txt';
        $llms_txt   = ABSPATH . 'llms.txt';

        // Get first 10 public pages/posts for the "first visible pages" table
        $visible_pages = get_posts([
            'post_type'      => ['post','page'],
            'post_status'    => 'publish',
            'posts_per_page' => 10,
            'orderby'        => 'date',
            'order'          => 'DESC',
        ]);

        // We'll also scan up to 50 public posts/pages to create Guidance and affected lists
        $scan_posts = get_posts([
            'post_type'      => ['post','page'],
            'post_status'    => 'publish',
            'posts_per_page' => 50,
            'orderby'        => 'date',
            'order'          => 'DESC',
            'fields'         => 'ids'
        ]);

        // Prepare issue buckets
        $schema_missing = [];
        $no_glossary_terms = [];
        $no_faq = [];
        $noindex_pages = [];

        // loop and detect issues (lightweight heuristics using local content)
        if (!empty($scan_posts)) {
            foreach ($scan_posts as $post_id) {
                // raw content (unfiltered) and filtered copy (for text checks)
                $raw = get_post_field('post_content', $post_id);
                $filtered_text = strtolower( strip_tags( apply_filters('the_content', $raw) ) );

                // 1. Schema presence (quick local heuristic)
                // If the post content itself contains application/ld+json or @type then mark as present locally
                $has_schema_local = ( strpos($raw, 'application/ld+json') !== false || stripos($raw, '"@type"') !== false );

                if (! $has_schema_local) {
                    $schema_missing[] = $post_id;
                }

                // 2. Glossary link presence heuristic
                if (! preg_match('/href=.*?\/glossary|class=.*?ss-glossary|class=.*?ss-glossary-term|glossary/i', $raw)) {
                    $no_glossary_terms[] = $post_id;
                }

                // 3. FAQ or Q&A (basic heuristics)
                if (! preg_match('/faq|frequently asked|q&amp;a|q&a|question:|answer:/i', $raw) && ! preg_match('/<div[^>]*class=["\'].*ss-qa.*["\']/i', $raw)) {
                    $no_faq[] = $post_id;
                }

                // 4. Indexability issues: look for noindex meta or Yoast meta
                $yoast_noindex = get_post_meta($post_id, '_yoast_wpseo_meta-robots-noindex', true);
                if ( preg_match('/noindex/i', $raw) || $yoast_noindex === '1' ) {
                    $noindex_pages[] = $post_id;
                }
            }
        }

        // counts
        $count_schema = count($schema_missing);
        $count_glossary = count($no_glossary_terms);
        $count_faq = count($no_faq);
        $count_noindex = count($noindex_pages);

        // Begin output
        echo '<div class="wrap" style="max-width:1100px;">';
        echo '<h1 style="margin-bottom:12px;">🧠 AI Visibility Checks (Free)</h1>';

        echo '<div class="ss-cta">';
        echo '👋 <strong>Welcome!</strong> This free dashboard shows how AI models (ChatGPT, Gemini, Claude, Perplexity) discover and read your website. Use the guidance below to make pages more quotable and visible to AI.';
        echo '</div>';

        /* --------------------------
           1 — AI ACCESS CHECK
        --------------------------- */
        echo '<div class="ss-section">';
        echo '<h2>🔍 AI Visibility Check</h2>';

        echo '<p class="ss-mini" style="margin-bottom:10px;">Your site exposes the following AI discovery entry points:</p>';
        echo '<ul style="line-height:1.8;font-size:14px;">';
        echo '<li>Robots.txt: <a href="' . esc_url($robots_url) . '" target="_blank" rel="noopener">' . esc_html($robots_url) . '</a></li>';
        echo '<li>ai.txt: ' . ( file_exists($ai_txt) ? '<span class="ss-badge-green">✔ Present</span>' : '<span class="ss-badge-red">✖ Missing</span>' ) . '</li>';
        echo '<li>llms.txt: ' . ( file_exists($llms_txt) ? '<span class="ss-badge-green">✔ Present</span>' : '<span class="ss-badge-red">✖ Missing</span>' ) . '</li>';
        echo '</ul>';
        echo '</div>';

        /* --------------------------
           2 — FIRST 10 PAGES VISIBLE
        --------------------------- */
        echo '<div class="ss-section">';
        echo '<h2>📄 First 10 Pages Visible to AI Engines</h2>';

        if (!empty($visible_pages)) {
            echo '<table class="widefat striped ss-table" style="margin-top:10px;">';
            echo '<thead><tr><th style="width:35%;">Page</th><th style="width:55%;">URL</th><th style="width:10%;">AI Status</th></tr></thead><tbody>';
            foreach ($visible_pages as $p) {
                $url = get_permalink($p->ID);
                echo '<tr>';
                echo '<td><strong>' . esc_html( $p->post_title ?: '(No Title)' ) . '</strong></td>';
                echo '<td><a href="' . esc_url($url) . '" target="_blank" rel="noopener">' . esc_html($url) . '</a></td>';
                echo '<td><span class="ss-badge-green">✔ Publicly Accessible</span></td>';
                echo '</tr>';
            }
            echo '</tbody></table>';
        } else {
            echo '<p class="ss-empty">No public pages found.</p>';
        }
        echo '</div>';

        /* --------------------------
           3 — GUIDANCE + AFFECTED PAGES
        --------------------------- */
        echo '<div class="ss-section">';
        echo '<h2>📘 What You Should Fix (Guidance)</h2>';
        echo '<p style="font-size:14px;line-height:1.6;">Improve your AI visibility by following these recommended actions:</p>';

        // short guidance box
        echo '<div style="background:#fbfcfd;border:1px solid #eef6fb;padding:12px;border-radius:6px;margin-bottom:12px;">';
        echo '<ul style="margin:0 0 0 18px;line-height:1.7;">';
        echo '<li><strong>Add Schema.org JSON-LD</strong> so AI can understand your content structure.</li>';
        echo '<li><strong>Link glossary terms</strong> to teach AI niche terminology.</li>';
        echo '<li><strong>Add Q&A / FAQ sections</strong> for better AI summarization.</li>';
        echo '<li><strong>Ensure pages are indexable</strong> (no noindex, no password, no robots block).</li>';
        echo '</ul>';
        echo '</div>';

   

        /* --------------------------
           FULL PAGE STATUS TABLE
        --------------------------- */
        echo '<div class="ss-section" style="margin-top:30px;">';
        echo '<h2>📋 Page-by-Page AI Status</h2>';

    echo '<table class="widefat striped ss-table" style="margin-top:10px;">';
echo '<thead><tr>
        <th>Page</th>
        <th>Schema</th>
        <th>Glossary</th>
        <th>FAQ</th>
        <th>Indexable</th>
        <th>Actions</th>
      </tr></thead><tbody>';

if (!empty($scan_posts)) {
    foreach ($scan_posts as $pid) {

        // use cached result or fetch freshly
        $result = get_transient(self::TRANSIENT_PREFIX . intval($pid));
        if ($result === false) {
            $result = self::detect_schema_status($pid);
            set_transient(self::TRANSIENT_PREFIX . intval($pid), $result, self::TRANSIENT_TTL);
        }

        $title    = esc_html(get_the_title($pid) ?: "(ID {$pid})");
        $edit_url = esc_url(get_edit_post_link($pid));
        $view_url = esc_url(get_permalink($pid));

        echo '<tr>';
        echo '<td><strong>' . $title . '</strong></td>';

        echo '<td>' . ( !empty($result['schema']) 
                ? '<span class="ss-badge-green">✔</span>' 
                : '<span class="ss-badge-red">✖</span>' ) . '</td>';

        echo '<td>' . ( !empty($result['glossary']) 
                ? '<span class="ss-badge-green">✔</span>' 
                : '<span class="ss-badge-red">✖</span>' ) . '</td>';

        echo '<td>' . ( !empty($result['faq']) 
                ? '<span class="ss-badge-green">✔</span>' 
                : '<span class="ss-badge-red">✖</span>' ) . '</td>';

        $is_indexable = ! in_array($pid, $noindex_pages);
        echo '<td>' . ( $is_indexable 
                ? '<span class="ss-badge-green">✔</span>' 
                : '<span class="ss-badge-red">✖</span>' ) . '</td>';

        // ACTIONS COLUMN – EDIT + VIEW ICON BUTTONS
        echo '<td>
                <a href="' . $edit_url . '" 
                    class="button-small ss-button-edit" 
                    title="Edit Page" 
                    target="_blank" 
                    rel="noopener"
                    style="margin-right:6px;">
                    ✏️ Edit
                </a>

                <a href="' . $view_url . '" 
                    class="button-small ss-button-edit" 
                    title="View Page" 
                    target="_blank" 
                    rel="noopener">
                    👁️ View
                </a>
              </td>';

        echo '</tr>';
    }
} else {
    echo '<tr><td colspan="6" class="ss-empty">No pages found for analysis.</td></tr>';
}

echo '</tbody></table>';
        echo '</div>'; // end full page status

        /* --------------------------
           4 — ONBOARDING
        --------------------------- */
        echo '<div class="ss-section">';
        echo '<h2>🚀 Onboarding Guide</h2>';
        echo '<ol style="line-height:1.7;">';
        echo '<li>Create your Glossary (define key concepts).</li>';
        echo '<li>Enable base schema for posts/pages.</li>';
        echo '<li>Review AI discovery signals on this page.</li>';
        echo '<li>Upgrade to Pro to unlock automated scanning & auto-fix.</li>';
        echo '</ol>';
        echo '</div>';

        /* --------------------------
           5 — SETUP PATH
        --------------------------- */
        echo '<div class="ss-section">';
        echo '<h2>🧭 Setup Path</h2>';
        echo '<ol style="line-height:1.7;">';
        echo '<li>Check ai.txt / llms.txt presence above.</li>';
        echo '<li>Review publicly visible URLs above.</li>';
        echo '<li>Add missing Schema + Glossary + Q&A where needed.</li>';
        echo '<li>Optionally upgrade to Pro for automation.</li>';
        echo '</ol>';
        echo '</div>';

        echo '</div>'; // wrap end
    }

    /**
     * Detect schema/glossary/faq presence for a single post.
     *
     * Strategy:
     * 1. Try to fetch final rendered HTML (remote GET) with short timeout;
     * 2. If remote fetch fails or returns error, fall back to local post content checks;
     * 3. Return an array: [ 'schema' => bool, 'faq' => bool, 'glossary' => bool, 'error' => bool ]
     */
    private static function detect_schema_status($post_id) {

        $permalink = get_permalink($post_id);
        if (! $permalink) {
            return [ 'schema' => false, 'faq' => false, 'glossary' => false, 'error' => true ];
        }

        // 1) try remote fetch with short timeout
        $args = [
            'timeout'   => 5,
            'redirection' => 3,
            'headers'   => [ 'User-Agent' => 'SearchShifter-FreeChecks/1.0 (+https://your-site.example)' ],
            'sslverify' => apply_filters('https_local_ssl_verify', true)
        ];

        $resp = wp_remote_get($permalink, $args);

        if (!is_wp_error($resp) && wp_remote_retrieve_response_code($resp) >= 200 && wp_remote_retrieve_response_code($resp) < 400) {
            $html = wp_remote_retrieve_body($resp);
            // basic checks in final rendered HTML
            $has_schema = ( stripos($html, 'application/ld+json') !== false || stripos($html, '"@type"') !== false );
            $has_faq    = ( stripos($html, 'faqpage') !== false || stripos($html, 'schema.org/Question') !== false || stripos($html, 'schema.org/FAQPage') !== false );
            $has_gloss  = ( preg_match('/\/glossary\//i', $html) > 0 || stripos($html, 'ss-glossary') !== false || stripos($html, 'ss-glossary-term') !== false );

            return [
                'schema'   => (bool) $has_schema,
                'faq'      => (bool) $has_faq,
                'glossary' => (bool) $has_gloss,
                'error'    => false
            ];
        }

        // 2) fallback: analyze local post content to provide a best-effort result
        $raw = get_post_field('post_content', $post_id);
        if ($raw === null) {
            return [ 'schema' => false, 'faq' => false, 'glossary' => false, 'error' => true ];
        }

        $has_schema_local = ( strpos($raw, 'application/ld+json') !== false || stripos($raw, '"@type"') !== false );
        $has_faq_local    = ( preg_match('/faq|frequently asked|q&a|question:|answer:/i', $raw) || preg_match('/<div[^>]*class=["\'].*ss-qa.*["\']/i', $raw) );
        $has_gloss_local  = ( preg_match('/href=.*?\/glossary|class=.*?ss-glossary|class=.*?ss-glossary-term|glossary/i', $raw) );

        return [
            'schema'   => (bool) $has_schema_local,
            'faq'      => (bool) $has_faq_local,
            'glossary' => (bool) $has_gloss_local,
            'error'    => false // not an error — just fallback
        ];
    }
}

// Initialize
SS_Free_Features::init();
