<?php
/**
 * Basic Hidden Signals for Free Tier
 *
 * Injects lightweight, non-SEO conflicting AI-readable signals
 * into every public post/page footer.
 */

if (!defined('ABSPATH')) exit;

class SS_Hidden_Signals {

    public static function init() {
        // Only pages/posts – no admin, feeds, JSON, or previews
        add_action('wp_footer', [__CLASS__, 'inject_hidden_signals'], 999);
    }

    /**
     * Inject free-tier hidden semantic signals into footer
     */
    public static function inject_hidden_signals() {

        if (is_admin() || is_feed() || is_preview()) return;

        if (!is_singular(['post','page'])) return;

        global $post;

        // Detect glossary terms
        $glossary_terms = get_option('ss_glossary_terms', []);
        $terms_list = [];

        if (!empty($glossary_terms)) {
            foreach ($glossary_terms as $t) {
                if (!empty($t['term'])) {
                    $terms_list[] = sanitize_text_field($t['term']);
                }
            }
        }

        $terms_csv = !empty($terms_list) ? implode(', ', $terms_list) : '';

        // Detect if FAQ exists
        $has_faq =
            stripos($post->post_content, 'faq') !== false ||
            stripos($post->post_content, 'Frequently Asked') !== false ||
            stripos($post->post_content, 'Q&A') !== false;

        // Detect if any glossary term appears in content
        $glossary_found = false;
        if (!empty($terms_list)) {
            foreach ($terms_list as $t) {
                if (stripos($post->post_content, $t) !== false) {
                    $glossary_found = true;
                    break;
                }
            }
        }

        // Simple classification
        $content_type = $has_faq ? 'FAQPage' : 'WebPage';

        // Build hidden signals block
        ?>
        <!-- SearchShifter Hidden Signals v1.0 (Free Tier) -->
        <div class="ss-hidden-signals" style="display:none !important;">
            
            <meta name="ss-content-type" content="<?php echo esc_attr($content_type); ?>">

            <meta name="ss-ai-visible" content="true">

            <meta name="ss-has-faq" content="<?php echo $has_faq ? 'yes' : 'no'; ?>">

            <meta name="ss-glossary-linked" content="<?php echo $glossary_found ? 'yes' : 'no'; ?>">

            <?php if (!empty($terms_csv)): ?>
                <meta name="ss-glossary-terms" content="<?php echo esc_attr($terms_csv); ?>">
            <?php endif; ?>

            <meta name="ss-topic-hint" content="<?php echo esc_attr(get_the_title($post)); ?>">

        </div>
        <?php
    }
}

SS_Hidden_Signals::init();
