<?php
/**
 * includes/schema-glossary.php
 *
 * Authority-ready dynamic glossary schema (single-file replacement).
 *
 * Replaces the legacy schema-glossary.php with a single, tier-aware schema output
 * that reads the ss_glossary CPT and outputs DefinedTerm + DefinedTermSet + site-level entities.
 */

if (! defined('ABSPATH')) {
    exit;
}

add_action('wp_head', function () {

    // -------- Governor toggle (use constant if available, otherwise option) -----
    $govern = true;
    if (defined('SS_Settings::OPT_SCHEMA_GOVERNOR')) {
        // can't reference constant string constant that way — fallback to option
        $govern = (bool) get_option( 'ss_schema_governor', true );
    } else {
        // prefer the plugin-defined opt constant if exists
        if ( defined('SS_Settings') && is_callable(['SS_Settings', 'class'])) {
            // noop — keep fallback — plugin may define SS_Settings class separately
        }
        $govern = (bool) get_option( SS_Settings::OPT_SCHEMA_GOVERNOR ?? 'ss_schema_governor', true );
    }

    if (! $govern) {
        return; // schema governor disabled
    }

    // -------- Only run on the Glossary page ---------------------------------
    $glossary_page_id = (int) get_option('ss_glossary_page_id');
 if (! $glossary_page_id || ! is_page($glossary_page_id) || is_singular('ss_glossary')) {
    return;
}


    // Disable other schema generators on this page (Yoast / Astra)
    // Use high priority so it's effective before output
    add_filter('wpseo_json_ld_output', '__return_empty_array', 99);
    add_filter('wpseo_schema_needs_to_output', '__return_false', 99);
    // Astra removal (best-effort)
    remove_action('astra_head', 'astra_add_schema');

    // -------- Load glossary CPT posts ---------------------------------------
    $terms = get_posts([
        'post_type'      => 'ss_glossary',
        'posts_per_page' => -1,
        'post_status'    => 'publish',
        'orderby'        => 'title',
        'order'          => 'ASC',
    ]);

    if (empty($terms)) {
        return; // nothing to output
    }

    $site_url     = trailingslashit(home_url());
    $glossary_url = get_permalink($glossary_page_id);
    $site_name    = get_bloginfo('name');
    $faq_page_id  = (int) get_option('ss_faq_page_id');
    $faq_url      = $faq_page_id ? get_permalink($faq_page_id) : $site_url . 'faq/';

    // Attempt to detect logo
    $logo_url = '';
    $custom_logo_id = get_theme_mod('custom_logo');
    if ($custom_logo_id) {
        $img = wp_get_attachment_image_src($custom_logo_id, 'full');
        if (!empty($img[0])) $logo_url = esc_url($img[0]);
    }
    if (!$logo_url) {
        // sensible fallback path (may not exist)
        $logo_url = trailingslashit($site_url) . 'wp-content/uploads/logo.png';
    }

    // Current site plan (if licensing class exists)
    $plan = 'free';
    if (class_exists('SS_Licensing')) {
        try {
            $license = SS_Licensing::get_cached_status();
            $plan = strtolower($license['plan'] ?? 'free');
        } catch (Throwable $e) {
            $plan = 'free';
        }
    }

    // Build DefinedTerm entries
    $defined_terms = [];
    foreach ($terms as $t) {

        $term_name = wp_strip_all_tags($t->post_title);
        $definition = wp_strip_all_tags(get_post_meta($t->ID, 'ss_definition', true) ?: $t->post_content ?: '');
        $same_as = get_post_meta($t->ID, 'ss_same_as', true) ?: '';
        $video   = get_post_meta($t->ID, 'ss_video_url', true) ?: '';
        $related = (array) get_post_meta($t->ID, 'ss_related_terms', true);
        $term_plan_flag = get_post_meta($t->ID, 'ss_plan_flag', true) ?: 'free';

        $entry = [
            '@type' => 'DefinedTerm',
            '@id'   => get_permalink($t->ID) . '#definedterm',
            'name'  => $term_name,
            'description' => $definition ?: '',
            'url'   => get_permalink($t->ID),
            'inDefinedTermSet' => $glossary_url,
        ];

        if (!empty($same_as)) {
            $entry['sameAs'] = esc_url_raw($same_as);
        }

        // PRO+ — expose related links as relatedLink for schema consumers
        if (!empty($related) && is_array($related) && in_array($plan, ['pro','authority','elite'], true)) {
            $links = [];
            foreach ($related as $rid) {
                $rid = intval($rid);
                if (!$rid) continue;
                if (!get_post_status($rid)) continue;
                $links[] = [
                    '@type' => 'DefinedTerm',
                    'name'  => get_the_title($rid),
                    '@id'   => get_permalink($rid) . '#definedterm'
                ];
            }
            if (!empty($links)) {
                $entry['relatedLink'] = $links;
            }
        }

        // AUTHORITY — include video multimodal signal
        if (!empty($video) && $plan === 'authority') {
            $entry['subjectOf'] = [
                '@type' => 'VideoObject',
                'url'   => esc_url_raw($video)
            ];
        }

        // Optionally include a 'termPlan' field to surface the term-level flag
        $entry['termPlan'] = $term_plan_flag;

        $defined_terms[] = $entry;
    }

    // Build the final graph
    $graph = [];

    // DefinedTermSet container
    $graph[] = [
        '@type' => 'DefinedTermSet',
        '@id'   => $glossary_url,
        'name'  => 'SearchShifter Glossary',
        'description' => 'Glossary of AI Visibility terms.',
        'url'   => $glossary_url,
        'hasDefinedTerm' => array_map(function($d){
            // reduce to reference objects for the set
            return [
                '@id' => $d['@id'],
                'name' => $d['name']
            ];
        }, $defined_terms)
    ];

    // Append all DefinedTerm entries
    foreach ($defined_terms as $dt) {
        $graph[] = $dt;
    }

    // Optional FAQ entity (if user configured FAQ page)
    if ($faq_page_id) {
        $graph[] = [
            '@type' => 'FAQPage',
            '@id'   => $faq_url . '#faq',
            'url'   => $faq_url,
            'name'  => 'FAQ — ' . $site_name
        ];
    }

    // Organization and WebSite
    $graph[] = [
        '@type' => 'Organization',
        '@id'   => $site_url . '#organization',
        'name'  => $site_name,
        'url'   => $site_url,
        'logo'  => [
            '@type' => 'ImageObject',
            'url' => $logo_url
        ],
    ];

    $graph[] = [
        '@type' => 'WebSite',
        '@id'   => $site_url . '#website',
        'url'   => $site_url,
        'name'  => $site_name,
        'publisher' => [ '@id' => $site_url . '#organization' ],
        'potentialAction' => [
            '@type' => 'SearchAction',
            'target' => $site_url . '?s={search_term_string}',
            'query-input' => 'required name=search_term_string'
        ]
    ];

    // BreadcrumbList for glossary
    $graph[] = [
        '@type' => 'BreadcrumbList',
        '@id'   => $glossary_url . '#breadcrumb',
        'itemListElement' => [
            [
                '@type' => 'ListItem',
                'position' => 1,
                'name' => 'Home',
                'item' => $site_url
            ],
            [
                '@type' => 'ListItem',
                'position' => 2,
                'name' => 'Glossary',
                'item' => $glossary_url
            ]
        ]
    ];

    $schema = [
        '@context' => 'https://schema.org',
        '@graph'   => $graph
    ];

    echo '<script type="application/ld+json">' .
         wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) .
         '</script>';

}, 99);
