<?php
if (!defined('ABSPATH')) exit;

/**
 * Bridges SearchShifter Organization settings into Yoast or Rank Math schema.
 * - Yoast: filters the Organization piece.
 * - Rank Math: walks the JSON-LD graph and overwrites Organization nodes.
 */
class SS_SEO_Bridge {

    public static function init() {
        // Fire late so the SEO plugins have already built their graph.
        add_filter('wpseo_schema_organization', [__CLASS__, 'yoast_org'], 99, 2);
        add_filter('rank_math/json_ld',          [__CLASS__, 'rankmath_graph'], 99, 2);
    }

    /** Read org settings with sensible defaults */
    private static function org_settings(): array {
        $defaults = [
            'name'        => get_bloginfo('name'),
            'url'         => home_url('/'),
            'logo_url'    => '',
            'email'       => get_option('admin_email'),
            'sitemap_url' => function_exists('wp_sitemaps_get_server') ? home_url('/wp-sitemap.xml') : home_url('/sitemap.xml'),
        ];
        $o = wp_parse_args((array) get_option('ss_org_sync', []), $defaults);

        // Build minimal structures that Yoast/Rank Math accept
        $logo = [];
        if (!empty($o['logo_url'])) {
            $logo = [
                '@type' => 'ImageObject',
                'url'   => esc_url_raw($o['logo_url']),
            ];
        }
        $contact = [];
        if (!empty($o['email'])) {
            $contact[] = [
                '@type'       => 'ContactPoint',
                'email'       => sanitize_email($o['email']),
                'contactType' => 'customer support',
            ];
        }

        return [
            'name'         => sanitize_text_field($o['name']),
            'url'          => esc_url_raw($o['url']),
            'logo'         => $logo,
            'contactPoint' => $contact,
        ];
    }

    /** Yoast: tweak the Organization piece */
    public static function yoast_org($data, $context) {
        $o = self::org_settings();

        // Keep Yoast's @id if present
        $out = is_array($data) ? $data : [];
        $out['@type'] = 'Organization';
        $out['name']  = $o['name'];
        $out['url']   = $o['url'];

        if (!empty($o['logo']))         $out['logo']         = $o['logo'];
        if (!empty($o['contactPoint'])) $out['contactPoint'] = $o['contactPoint'];

        return $out;
    }

    /** Rank Math: overwrite all Organization nodes in the graph */
    public static function rankmath_graph($graph, $jsonld) {
        if (!is_array($graph)) return $graph;
        $o = self::org_settings();

        foreach ($graph as $i => $node) {
            $types = isset($node['@type']) ? (array) $node['@type'] : [];
            if (in_array('Organization', $types, true)) {
                $node['name'] = $o['name'];
                $node['url']  = $o['url'];
                if (!empty($o['logo']))         $node['logo']         = $o['logo'];
                if (!empty($o['contactPoint'])) $node['contactPoint'] = $o['contactPoint'];
                $graph[$i] = $node;
            }
        }
        return $graph;
    }
}
