<?php
if (!defined('ABSPATH')) exit;

class SS_Settings {
    const OPT_GROUP         = 'searchshifter';

    const OPT_API_KEY       = 'ss_api_key';
    const OPT_DOMAIN        = 'ss_domain';
    const OPT_API_BASE      = 'ss_api_base';

    // Signing
    const OPT_SIGNING_METHOD= 'ss_signing_method';    // 'hmac' or 'ed25519'
    const OPT_SIGNING_SECRET= 'ss_signing_secret';    // for HMAC
    const OPT_PUBLIC_KEY    = 'ss_public_key_b64';    // optional Ed25519 pubkey (base64)

    // Features & privacy
    const OPT_FEATURE_FLAGS = 'ss_feature_flags';
    const OPT_TELEMETRY     = 'ss_telemetry_opt_in';

    const OPT_SCHEMA_GOVERNOR = 'ss_schema_governor'; // new global schema toggle

    public static function init() {
        add_action('admin_menu',  [__CLASS__,'menu']);
        add_action('admin_init',  [__CLASS__,'register']);
        add_action('admin_notices', [__CLASS__,'license_notice']);
        add_action('network_admin_notices',[__CLASS__, 'license_notice']); // optional
        // When feature flags change, flush rewrites + clear cached coach result
        add_action('update_option_' . self::OPT_FEATURE_FLAGS, [__CLASS__, 'on_flags_updated'], 10, 2);
        add_action('admin_post_ss_refresh_license', [__CLASS__, 'handle_refresh_license']);

    }

        public static function handle_refresh_license(){
            if (!current_user_can('manage_options')) {
                wp_die('No permission');
            }

            check_admin_referer('ss_refresh_license');

            // ✅ REAL-TIME CHECK (no cache, no force refresh)
        // ✅ FORCE fresh license check (no cache)
        $st = class_exists('SS_Licensing')
            ? SS_Licensing::force_refresh()
            : ['active' => false, 'message' => 'Licensing class missing'];


            if (($st['status'] ?? '') === 'free') {
                SS_Admin_Notices::success('Free plan active.');
            } elseif (!empty($st['active'])) {
                SS_Admin_Notices::success('License verified successfully.');
            } else {
                SS_Admin_Notices::warning($st['message'] ?: 'License not active.');
            }

            wp_safe_redirect(
                wp_get_referer() ?: admin_url('admin.php?page=searchshifter')
            );
            exit;
        }




    /** If flags change, clear caches and refresh rewrites (for /llms.txt). */
    public static function on_flags_updated($old, $new){
        $old = (array) $old; $new = (array) $new;
        $had_llms = in_array('llms', $old, true);
        $has_llms = in_array('llms', $new, true);

        // bust the cached visibility score so the UI refreshes
        if (class_exists('SS_Visibility')) {
            delete_transient(SS_Visibility::TRANSIENT_KEY);
        } else {
            delete_transient('ss_last_visibility_result');
        }
        set_transient('ss_visibility_needs_rerun', 1, MINUTE_IN_SECONDS);

        if ($had_llms !== $has_llms) {
            flush_rewrite_rules();
        }
    }

    public static function menu() {
        add_menu_page(
            'SearchShifter',
            'SearchShifter',
            'manage_options',
            'searchshifter',
            [__CLASS__,'render'],
            'dashicons-search',
            65
        );

        add_submenu_page(
            'searchshifter',
            'AI Protocol Files',
            'AI Protocol Files',
            'manage_options',
            'searchshifter-ai',
            ['SS_AI_Admin','render']
        );

    }

    public static function register() {
        // ----- core connection -----
        register_setting(self::OPT_GROUP, self::OPT_API_KEY,
            ['type'=>'string','sanitize_callback'=>'sanitize_text_field']);
        register_setting(self::OPT_GROUP, self::OPT_DOMAIN,
            ['type'=>'string','sanitize_callback'=>'sanitize_text_field']);

        // enforce /api at the end
        register_setting(self::OPT_GROUP, self::OPT_API_BASE, [
            'type'              => 'string',
            'sanitize_callback' => [__CLASS__, 'sanitize_api_base'],
        ]);

        // ----- signing / crypto -----
        register_setting(self::OPT_GROUP, self::OPT_SIGNING_METHOD,
            ['type'=>'string','default'=>'hmac']);
        register_setting(self::OPT_GROUP, self::OPT_SIGNING_SECRET,
            ['type'=>'string','sanitize_callback'=>'trim']);
        register_setting(self::OPT_GROUP, self::OPT_PUBLIC_KEY,
            ['type'=>'string','sanitize_callback'=>'trim']);

        // ----- features & privacy -----
        register_setting(self::OPT_GROUP, self::OPT_FEATURE_FLAGS, [
            'type'=>'array','default'=>[],
            'sanitize_callback'=>[__CLASS__,'sanitize_flags']
        ]);
        register_setting(self::OPT_GROUP, self::OPT_TELEMETRY,
            ['type'=>'boolean','default'=>false]);

            // Global schema governor (Yoast/RankMath conflict prevention)
        register_setting(self::OPT_GROUP, self::OPT_SCHEMA_GOVERNOR, [
            'type' => 'boolean',
            'default' => true,
        ]);

        register_setting(self::OPT_GROUP, 'ss_org_sync', [
        'type' => 'array',
        'sanitize_callback' => [__CLASS__, 'sanitize_org'],]);

        // Sections / fields (let WP print the <label>; our callbacks print only inputs)
        add_settings_section('ss_conn', __('Connection','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field(self::OPT_DOMAIN,   __('Domain','searchshifter'),       [__CLASS__,'field_domain'],   self::OPT_GROUP, 'ss_conn', ['label_for'=>self::OPT_DOMAIN]);
        add_settings_field(self::OPT_API_KEY,  __('API Key','searchshifter'),      [__CLASS__,'field_api'],      self::OPT_GROUP, 'ss_conn', ['label_for'=>self::OPT_API_KEY]);
        add_settings_field(self::OPT_API_BASE, __('API Base URL','searchshifter'), [__CLASS__,'field_api_base'], self::OPT_GROUP, 'ss_conn', ['label_for'=>self::OPT_API_BASE]);

        add_settings_section('ss_sign', __('Signing & Verification','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field(self::OPT_SIGNING_METHOD, __('Signing Method','searchshifter'), [__CLASS__,'field_sign_method'], self::OPT_GROUP, 'ss_sign', ['label_for'=>self::OPT_SIGNING_METHOD]);
        add_settings_field(self::OPT_SIGNING_SECRET, __('Signing Secret (for HMAC)','searchshifter'), [__CLASS__,'field_sign_secret'], self::OPT_GROUP, 'ss_sign', ['label_for'=>self::OPT_SIGNING_SECRET]);

        add_settings_section('ss_crypto', __('Public Key (Ed25519, optional)','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field(self::OPT_PUBLIC_KEY, __('Public Key (base64)','searchshifter'), [__CLASS__,'field_pubkey'], self::OPT_GROUP, 'ss_crypto', ['label_for'=>self::OPT_PUBLIC_KEY]);

        add_settings_section('ss_flags', __('Feature Flags','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field(self::OPT_FEATURE_FLAGS, __('Flags','searchshifter'), [__CLASS__,'field_flags'], self::OPT_GROUP, 'ss_flags');

        add_settings_section('ss_priv', __('Privacy & Telemetry','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field(self::OPT_TELEMETRY, __('Allow Telemetry (opt-in)','searchshifter'), [__CLASS__,'field_telemetry'], self::OPT_GROUP, 'ss_priv');
        
        add_settings_section(
            'ss_schema_control',
            __('Schema Control','searchshifter'),
            '__return_false',
            self::OPT_GROUP
        );

        add_settings_field(
            self::OPT_SCHEMA_GOVERNOR,
            __('Use SearchShifter as Primary Schema Source','searchshifter'),
            [__CLASS__,'field_schema_governor'],
            self::OPT_GROUP,
            'ss_schema_control'
        );

        add_settings_section('ss_org', __('Organization','searchshifter'), '__return_false', self::OPT_GROUP);
        add_settings_field('ss_org_name',   __('Organization Name','searchshifter'), [__CLASS__,'field_org_name'],   self::OPT_GROUP, 'ss_org');
        add_settings_field('ss_org_url',    __('Organization URL','searchshifter'),  [__CLASS__,'field_org_url'],    self::OPT_GROUP, 'ss_org');
        add_settings_field('ss_org_logo',   __('Logo URL','searchshifter'),          [__CLASS__,'field_org_logo'],   self::OPT_GROUP, 'ss_org');
        add_settings_field('ss_org_email',  __('Contact Email','searchshifter'),     [__CLASS__,'field_org_email'],  self::OPT_GROUP, 'ss_org');
        add_settings_field('ss_org_map',    __('Sitemap URL','searchshifter'),       [__CLASS__,'field_org_map'],    self::OPT_GROUP, 'ss_org');

    }

    public static function field_schema_governor() {
    $id = self::OPT_SCHEMA_GOVERNOR;
    $on = (bool) get_option($id, true);
    ?>

    <!-- Hidden fallback ensures OFF saves correctly -->
    <input type="hidden" name="<?php echo esc_attr($id); ?>" value="0" />

    <label class="ss-toggle">
        <input type="checkbox"
            id="<?php echo esc_attr($id); ?>"
            name="<?php echo esc_attr($id); ?>"
            value="1"
            <?php checked($on, true); ?>
        />
        <span class="ss-toggle-slider"></span>
        <span class="ss-toggle-label">Enable SearchShifter Schema Governor</span>
    </label>

    <p class="ss-help">
        <strong>User Facing Explanation:</strong><br>
        SearchShifter manages your schema so your site sends one clear signal to Google and AI engines.
        This avoids duplicate or conflicting schema often created by Yoast or RankMath.
    </p>

    <p class="ss-help">
        <strong>Agency Facing Explanation:</strong><br>
        SearchShifter acts as your schema governor for consistent structured data output.
        This eliminates mixed schema sources and improves clarity for Google & AI engines.
    </p>

    <?php
}




    public static function sanitize_api_base($url){
        $url = esc_url_raw(trim((string)$url));
        if ($url) $url = rtrim($url, '/');
        if ($url && !preg_match('#/api$#', $url)) {
            add_settings_error(self::OPT_API_BASE, 'ss_api_base', __('API Base must end with /api','searchshifter'), 'error');
        }
        return $url;
    }

    /** Public getter for Organization settings (merged with defaults, sanitized). */
    public static function get_org(): array {
        // Start with the same defaults the fields use
        $defaults = self::org_defaults();

        // Merge saved values over defaults
        $raw = (array) get_option('ss_org_sync', []);
        $o   = wp_parse_args($raw, $defaults);

        // Re-sanitize (mirrors sanitize_org)
        $out = [];
        $out['name']        = sanitize_text_field($o['name'] ?? $defaults['name']);
        $out['url']         = esc_url_raw(trim((string) ($o['url'] ?? $defaults['url'])));
        $out['logo_url']    = esc_url_raw(trim((string) ($o['logo_url'] ?? '')));
        $out['email']       = sanitize_email($o['email'] ?? $defaults['email']);
        $out['sitemap_url'] = esc_url_raw(trim((string) ($o['sitemap_url'] ?? $defaults['sitemap_url'])));

        return $out;
    }

    public static function sanitize_org($val){
        $out = [];
        $out['name']       = sanitize_text_field( $val['name']  ?? get_bloginfo('name') );
        $out['url']        = esc_url_raw( trim((string)($val['url'] ?? home_url('/'))) );
        $out['logo_url']   = esc_url_raw( trim((string)($val['logo_url'] ?? '')) );
        $out['email']      = sanitize_email( $val['email'] ?? get_option('admin_email') );
        $out['sitemap_url']= esc_url_raw( trim((string)($val['sitemap_url'] ?? '')) );
        return $out;
    }

    private static function org_defaults(){
        if (class_exists('SS_AI_Protocol') && is_callable(['SS_AI_Protocol','detect_sitemap_url'])) {
            $map = SS_AI_Protocol::detect_sitemap_url();
        } else {
            if (defined('WPSEO_VERSION') || class_exists('RankMath')) {
                $map = home_url('/sitemap_index.xml');
            } elseif (function_exists('wp_sitemaps_get_server')) {
                $map = home_url('/wp-sitemap.xml');
            } else {
                $map = home_url('/sitemap.xml');
            }
        }

        $logo = '';
        $id = (int) get_theme_mod('custom_logo');
        if ($id) {
            $src = wp_get_attachment_image_src($id, 'full');
            if (!empty($src[0])) $logo = $src[0];
        } elseif ( get_site_icon_url(512) ) {
            $logo = get_site_icon_url(512);
        }

        return [
            'name'        => get_bloginfo('name'),
            'url'         => home_url('/'),
            'logo_url'    => $logo,
            'email'       => get_option('admin_email'),
            'sitemap_url' => $map,
        ];
    }



    public static function field_org_name(){
        $o = wp_parse_args( (array) get_option('ss_org_sync', []), self::org_defaults() );
        printf('<input type="text" class="regular-text" name="ss_org_sync[name]" value="%s" />', esc_attr($o['name']));
    }

    public static function field_org_url(){
        $o = wp_parse_args( (array) get_option('ss_org_sync', []), self::org_defaults() );
        printf('<input type="url" class="regular-text code" name="ss_org_sync[url]" value="%s" />', esc_attr($o['url']));
    }

    public static function field_org_logo(){
        $o = wp_parse_args( (array) get_option('ss_org_sync', []), self::org_defaults() );
        printf('<input type="url" class="regular-text code" name="ss_org_sync[logo_url]" value="%s" placeholder="https://example.com/logo.png" />', esc_attr($o['logo_url']));
        echo '<p class="ss-help">'.esc_html__('Leave empty to auto-use the Custom Logo or Site Icon.','searchshifter').'</p>';
    }

    public static function field_org_email(){
        $o = wp_parse_args( (array) get_option('ss_org_sync', []), self::org_defaults() );
        printf('<input type="email" class="regular-text" name="ss_org_sync[email]" value="%s" />', esc_attr($o['email']));
    }

    public static function field_org_map(){
        $o = wp_parse_args( (array) get_option('ss_org_sync', []), self::org_defaults() );
        printf('<input type="url" class="regular-text code" name="ss_org_sync[sitemap_url]" value="%s" />', esc_attr($o['sitemap_url']));
    }

    public static function sanitize_flags($value){
        if (!is_array($value)) $value = [];
        $value = array_filter($value, fn($v) => $v !== '__none');
        $allowed = ['qa','llms','schema'];
        $out = [];
        foreach ($value as $v) {
            if (in_array($v, $allowed, true)) $out[] = $v;
        }
        return array_values(array_unique($out));
    }

    public static function render() {
        if (!current_user_can('manage_options')) return;

        // Load inline CSS/JS once
        self::inline_assets();

        echo '<div id="searchshifter" class="wrap">';

        /* ======================================================
        * Header
        * ====================================================== */
        $ui = SS_Licensing::ui_state();

        $badgeClass = $ui['badge']['class']; // ok | warn | err
        $badgeText  = $ui['badge']['text'];
        $lic_msg    = trim((string) ($ui['message'] ?? ''));

        $map = [
            'ok'   => 'ss-badge--ok',
            'warn' => 'ss-badge--warn',
            'err'  => 'ss-badge--err',
        ];
        $cls = $map[$badgeClass] ?? 'ss-badge--err';

        echo '<h1>SearchShifter Settings 
            <span class="ss-badge '.$cls.'">License: '.esc_html($badgeText).'</span>
        </h1>';

        /* ======================================================
        * Optional notice (visibility refresh)
        * ====================================================== */
        if (get_transient('ss_visibility_needs_rerun')) {
            echo '<div class="notice notice-warning"><p>
                Feature flags changed. Re-run the Visibility Score to refresh results.
            </p></div>';
            delete_transient('ss_visibility_needs_rerun');
        }

        /* ======================================================
        * Settings Form
        * ====================================================== */
        echo '<div class="ss-card">';
        echo '<form method="post" action="options.php">';
        settings_fields(self::OPT_GROUP);
        do_settings_sections(self::OPT_GROUP);
        submit_button();
        echo '</form>';
        echo '</div>';

        /* ======================================================
        * License Card (SINGLE RENDER)
        * ====================================================== */
        echo '<div class="ss-card">';
        echo '<h2 style="margin-top:0">License</h2>';

        echo '<p><span class="ss-badge '.$cls.'">'.esc_html($badgeText).'</span></p>';

        if ($lic_msg !== '') {
            echo '<p><em>' . esc_html($lic_msg) . '</em></p>';
        }

        if (!empty($ui['plan'])) {
            echo '<p><code>Plan:</code> ' . esc_html($ui['plan']) . '</p>';
        }

        if (!empty($ui['expires'])) {
            echo '<p><code>Expires:</code> ' . esc_html($ui['expires']) . '</p>';
        }

        // Re-check license button
        echo '<form method="post" action="'.esc_url(admin_url('admin-post.php')).'" style="margin-top:12px;">';
        wp_nonce_field('ss_refresh_license');
        echo '<input type="hidden" name="action" value="ss_refresh_license" />';
        echo '<button class="button button-secondary">Re-check License</button>';
        echo '</form>';

        echo '</div>'; // ss-card (License)

        echo '</div>'; // .wrap
    }


    // ---------- field renderers (NO duplicate labels here) ----------
    public static function field_domain(){
    $id = self::OPT_DOMAIN;
    $v  = get_option($id, parse_url(get_site_url(), PHP_URL_HOST));
    echo '<div class="ss-field">';
    echo '  <div class="ss-inputwrap">';
    printf('<input id="%s" type="text" class="regular-text" name="%s" value="%s" placeholder="example.com" />',
           esc_attr($id), esc_attr($id), esc_attr($v));
    printf('<button type="button" class="button ss-icon-btn ss-btn-copy" data-copy="#%s" title="%s"><span class="dashicons dashicons-clipboard"></span></button>',
           esc_attr($id), esc_attr__('Copy','searchshifter'));
    echo '  </div>';
    echo '  <p class="ss-help">'.esc_html__('Used to validate your license and scope API calls.','searchshifter').'</p>';
    echo '</div>';
    }

    // public static function field_api(){
    //     $id = self::OPT_API_KEY;
    //     $v  = get_option($id, '');
    //     echo '<div class="ss-field">';
    //     echo '  <div class="ss-inputwrap">';
    //     printf('<input id="%s" type="password" class="regular-text" name="%s" value="%s" placeholder="sk_live_..." autocomplete="new-password" />',
    //         esc_attr($id), esc_attr($id), esc_attr($v));
    //     printf('<button type="button" class="button ss-icon-btn ss-btn-toggle" data-toggle="#%s" title="%s"><span class="dashicons dashicons-visibility"></span></button>',
    //         esc_attr($id), esc_attr__('Show / hide','searchshifter'));
    //     printf('<button type="button" class="button ss-icon-btn ss-btn-copy" data-copy="#%s" title="%s"><span class="dashicons dashicons-clipboard"></span></button>',
    //         esc_attr($id), esc_attr__('Copy','searchshifter'));
    //     echo '  </div>';
    //     echo '  <p class="ss-help">'.esc_html__('Paste the key from your Thrivx dashboard.','searchshifter').'</p>';
    //     echo '</div>';
    // }

  public static function field_api() {

    $id = self::OPT_API_KEY;
    $v  = get_option($id, '');

    echo '<div class="ss-field">';
    echo '  <div class="ss-inputwrap">';

    printf(
        '<input id="%s" type="password" class="regular-text" name="%s" value="%s"
            placeholder="sk_live_..." autocomplete="new-password" />',
        esc_attr($id),
        esc_attr($id),
        esc_attr($v)
    );

    printf(
        '<button type="button" class="button ss-icon-btn ss-btn-toggle" data-toggle="#%s">
            <span class="dashicons dashicons-visibility"></span>
        </button>',
        esc_attr($id)
    );

    printf(
        '<button type="button" class="button ss-icon-btn ss-btn-copy" data-copy="#%s">
            <span class="dashicons dashicons-clipboard"></span>
        </button>',
        esc_attr($id)
    );

    echo '  </div>';

    echo '<p class="ss-help">';
    echo esc_html__('Enter your API key to activate Pro or Authority features.','searchshifter');
    echo '</p>';

    echo '</div>';
}



    public static function field_api_base(){
        $id = self::OPT_API_BASE;
        $default = defined('SEARCHSHIFTER_API_BASE') ? rtrim(SEARCHSHIFTER_API_BASE,'/') : 'http://127.0.0.1:8000/api';
        $v  = get_option($id, $default);
        echo '<div class="ss-field">';
        echo '  <div class="ss-inputwrap">';
        printf('<input id="%s" type="url" class="regular-text code" name="%s" value="%s" placeholder="https://yourdomain.com/path/api" />',
            esc_attr($id), esc_attr($id), esc_attr($v));
        printf('<button type="button" class="button ss-icon-btn ss-btn-copy" data-copy="#%s" title="%s"><span class="dashicons dashicons-clipboard"></span></button>',
            esc_attr($id), esc_attr__('Copy','searchshifter'));
        echo '  </div>';
        echo '  <p class="ss-help">'.esc_html__('Must end with','searchshifter').' <code>/api</code>.</p>';
        echo '</div>';
    }


    public static function field_sign_method(){
        $id = self::OPT_SIGNING_METHOD;
        $v  = get_option($id, 'hmac');
        echo '<div>';
        echo '<select id="'.esc_attr($id).'" name="'.esc_attr($id).'">';
        echo '<option value="hmac" '.selected($v,'hmac',false).'>HMAC-SHA256 (Laravel ResponseSigner)</option>';
        echo '<option value="ed25519" '.selected($v,'ed25519',false).'>Ed25519 (public key)</option>';
        echo '</select>';
        echo '<p class="ss-help">'.esc_html__('We verify server responses before saving schema.','searchshifter').'</p>';
        echo '</div>';
    }

    public static function field_sign_secret(){
        $id = self::OPT_SIGNING_SECRET;
        $v  = get_option($id, '');
        echo '<div class="ss-field">';
        echo '  <div class="ss-inputwrap">';
        printf('<input id="%s" type="password" class="regular-text" name="%s" value="%s" autocomplete="new-password" />',
            esc_attr($id), esc_attr($id), esc_attr($v));
        printf('<button type="button" class="button ss-icon-btn ss-btn-toggle" data-toggle="#%s" title="%s"><span class="dashicons dashicons-visibility"></span></button>',
            esc_attr($id), esc_attr__('Show / hide','searchshifter'));
        printf('<button type="button" class="button ss-icon-btn ss-btn-copy" data-copy="#%s" title="%s"><span class="dashicons dashicons-clipboard"></span></button>',
            esc_attr($id), esc_attr__('Copy','searchshifter'));
        echo '  </div>';
        echo '  <p class="ss-help">'.esc_html__('Required if you use HMAC signing.','searchshifter').'</p>';
        echo '</div>';
    }
    

    public static function field_pubkey(){
        $id = self::OPT_PUBLIC_KEY;
        $v  = get_option($id, '');
        printf('<textarea id="%s" name="%s" rows="3" class="large-text code">%s</textarea>', esc_attr($id), esc_attr($id), esc_textarea($v));
        echo '<p class="ss-help">'.esc_html__('Optional, for Ed25519 signature verification.','searchshifter').'</p>';
    }

    public static function field_flags(){
        $flags = (array) get_option(self::OPT_FEATURE_FLAGS, []);
        $known = ['qa'=>'Q&A CPT','llms'=>'llms.txt endpoint','schema'=>'Schema publish'];

        // ensure clearing works when all unchecked
        printf('<input type="hidden" name="%s[]" value="__none" />', esc_attr(self::OPT_FEATURE_FLAGS));

        foreach ($known as $key=>$label){
            $checked = in_array($key, $flags, true) ? 'checked' : '';
            printf(
                '<label style="display:block;margin:4px 0"><input type="checkbox" name="%s[]" value="%s" %s> %s</label>',
                esc_attr(self::OPT_FEATURE_FLAGS),
                esc_attr($key),
                $checked,
                esc_html($label)
            );
        }
        echo '<p class="ss-help">'.esc_html__('Toggling may flush routes (e.g., /llms.txt) and refresh visibility cache.','searchshifter').'</p>';
    }

    public static function field_telemetry(){
        $id = self::OPT_TELEMETRY;
        $on = (bool) get_option($id, false);
        printf('<label><input id="%s" type="checkbox" name="%s" value="1" %s /> %s</label>',
            esc_attr($id), esc_attr($id), checked($on, true, false),
            esc_html__('Share anonymous diagnostics only (no PII, no content).','searchshifter')
        );
    }

    public static function license_notice(){
        if ( ! current_user_can('manage_options')) return;
        // if ( ! function_exists('get_current_screen')) return;

        //$screen = get_current_screen();
        // Only show on the SearchShifter settings screen
        //if ( ! $screen || strpos($screen->id, 'searchshifter') === false ) return;

        $status       = class_exists('SS_Licensing') ? SS_Licensing::get_cached_status() : ['active' => false];
        $settings_url = admin_url('admin.php?page=searchshifter');

        // Only show the banner when inactive
        $status = SS_Licensing::get_cached_status();

        // ✅ Do NOT show warning for Free users
        if (($status['status'] ?? '') === 'free') {
            return;
        }

        // Only warn if truly inactive
        if (!empty($status['active'])) {
            return;
        }


        echo '<div class="notice notice-warning is-dismissible">';
        echo '<div class="ss-license-box">';

            // Left side: message
            echo '<div class="ss-license-text"><strong>SearchShifter:</strong> License inactive. The plugin runs in <em>read-only</em> mode.</div>';

            // Right side: actions
            echo '<div class="ss-license-actions">';
            echo '<a class="button button-primary" href="' . esc_url($settings_url) . '">Open Settings</a>';

            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
                wp_nonce_field('ss_refresh_license');
                echo '<input type="hidden" name="action" value="ss_refresh_license" />';
                echo '<button class="button" type="submit">Re-check license</button>';
            echo '</form>';

            if ( ! empty($status['message']) ) {
                echo '<span class="ss-license-msg"><em>' . esc_html($status['message']) . '</em></span>';
            }
            echo '</div>'; // .ss-license-actions

        echo '</div>'; // .ss-license-box

        // Scoped CSS for alignment (does NOT touch .notice globally)
        echo '<style>
            .ss-license-box{display:flex;align-items:center;justify-content:space-between;gap:16px}
            .ss-license-text{margin:6px 0}
            .ss-license-actions{display:flex;align-items:center;gap:8px;margin:6px 0}
            .ss-license-actions form{margin:0}
            .ss-license-msg{color:#646970}
            @media (max-width:782px){.ss-license-box{flex-direction:column;align-items:flex-start}}
        </style>';
        echo '</div>';
    }



    /** Tiny CSS/JS for icons + layout (no external assets). */
    private static function inline_assets(){
    ?>
    <style>
        /* --- Card + badge (unchanged) --- */
        #searchshifter .ss-card{
        background:#fff;border:1px solid #dcdcde;border-radius:6px;padding:16px 18px;margin:18px 0;
        }
        #searchshifter .ss-badge{display:inline-block;padding:3px 8px;border-radius:999px;font-size:12px;vertical-align:middle;margin-left:8px}
        #searchshifter .ss-badge--ok{background:#e7f7ec;color:#006e2b;border:1px solid #b7e4c6}
        #searchshifter .ss-badge--err{background:#fdeaea;color:#8a1f11;border:1px solid #f5c2c0}
        #searchshifter .ss-badge--warn{background:#fff8e5;color:#7a5;border:1px solid #f7e1a5}

        .ss-field{margin-bottom:12px}
        .ss-help{color:#666;margin:4px 0 0}

        /* --- NEW: icons outside the input --- */
        #searchshifter .ss-inputwrap{
        display:flex;            /* row: input then icons */
        align-items:center;
        gap:6px;                 /* space between input and icons */
        max-width:520px;
        }

        #searchshifter .ss-inputwrap input.regular-text{
        flex:1;                  /* input takes remaining width */
        padding-right:12px;      /* normal padding (no extra for overlay icons) */
        }

        /* Compact icon buttons (no WP button chrome) */
        #searchshifter .button.ss-icon-btn{
        position:static;         /* NO absolute positioning */
        min-width:0;
        width:24px; height:24px; /* smaller */
        padding:0; margin:0;
        border:none; background:transparent; box-shadow:none;
        display:flex; align-items:center; justify-content:center;
        border-radius:4px; cursor:pointer; color:#50575e;
        }
        #searchshifter .button.ss-icon-btn .dashicons{
        font-size:18px; width:18px; height:18px; line-height:18px;
        }

        #searchshifter .button.ss-icon-btn:hover{background:#f0f0f1;color:#1d2327}
        #searchshifter .button.ss-icon-btn:focus{outline:2px solid #2271b1;outline-offset:1px}

        /* If you want a hair more spacing to the right of the icons on tight screens */
        @media (max-width: 782px){
        #searchshifter .ss-inputwrap{gap:8px}
        }

        /* ===== Visibility panel polish ===== */
        #searchshifter .ss-vrow{
        display:flex; align-items:center; gap:8px; max-width:560px;
        }
        #searchshifter .ss-vrow input[type="url"]{ flex:1; }

        #searchshifter .ss-score{
        display:inline-flex; align-items:center; gap:6px;
        padding:2px 10px; border-radius:999px; font-weight:600; font-size:12px;
        border:1px solid transparent;
        }
        #searchshifter .ss-score--good{ background:#e7f7ec; color:#0a5; border-color:#bfe8cf; }
        #searchshifter .ss-score--ok  { background:#fff8e5; color:#7a5; border-color:#f7e1a5; }
        #searchshifter .ss-score--bad { background:#fdeaea; color:#a33; border-color:#f5c2c0; }

        #searchshifter .ss-url-inline{
        margin-left:10px; font-size:12px;
        }
        #searchshifter .ss-url-inline a{ color:#2271b1; text-decoration:none; }
        #searchshifter .ss-url-inline a:hover{ text-decoration:underline; }

        /* Signal “chips” */
        #searchshifter .ss-signal-list{ margin:8px 0 0; padding:0; display:flex; flex-wrap:wrap; gap:6px; }
        #searchshifter .ss-chip{
        display:inline-block; padding:2px 8px; font-size:12px; border-radius:999px;
        border:1px solid #dcdcde; background:#f6f7f7; color:#1d2327;
        }
        #searchshifter .ss-chip--good{ background:#eefaf1; border-color:#cdeed7; color:#11623a; }
        #searchshifter .ss-chip--warn{ background:#fff7e6; border-color:#f4e2b2; color:#7a5a12; }
        #searchshifter .ss-chip--bad { background:#fdeaea; border-color:#f5c2c0; color:#8a1f11; }

        /* Recommendations list */
        #searchshifter .ss-recs{ margin:6px 0 0 18px; list-style:disc; }

/* --- Toggle Switch --- */
.ss-toggle {
    position: relative;
    display: flex;
    align-items: center;
    gap: 10px;
    cursor: pointer;
    user-select: none;
}

.ss-toggle input {
    display: none;
}

.ss-toggle-slider {
    position: relative;
    width: 44px;
    height: 22px;
    background: #ccc;
    border-radius: 22px;
    transition: background 0.25s;
}

.ss-toggle-slider::before {
    content: "";
    position: absolute;
    width: 18px;
    height: 18px;
    left: 2px;
    top: 2px;
    border-radius: 50%;
    background: #fff;
    box-shadow: 0 1px 2px rgba(0,0,0,0.2);
    transition: transform 0.25s;
}

.ss-toggle input:checked + .ss-toggle-slider {
    background: #2271b1;
}

.ss-toggle input:checked + .ss-toggle-slider::before {
    transform: translateX(22px);
}

.ss-toggle-label {
    font-size: 14px;
}

    </style>

    <script>
    (function(){
        function qs(s,root){return (root||document).querySelector(s);}

        function bindHandlers(){
            // COPY
            document.addEventListener('click', function(e){
                var btn = e.target.closest('.ss-btn-copy');
                if(!btn) return;
                var sel = btn.getAttribute('data-copy');
                var el  = qs(sel);
                if(!el) return;
                navigator.clipboard.writeText(el.value || '').then(function(){
                    var ic = btn.querySelector('.dashicons');
                    if(!ic) return;
                    ic.classList.remove('dashicons-clipboard');
                    ic.classList.add('dashicons-yes');
                    setTimeout(function(){
                        ic.classList.remove('dashicons-yes');
                        ic.classList.add('dashicons-clipboard');
                    }, 900);
                });
            });

            // TOGGLE show/hide
            document.addEventListener('click', function(e){
                var btn = e.target.closest('.ss-btn-toggle');
                if(!btn) return;
                var sel = btn.getAttribute('data-toggle');
                var el  = qs(sel);
                if(!el) return;
                var to  = el.type === 'password' ? 'text' : 'password';
                el.type = to;
                var ic = btn.querySelector('.dashicons');
                if(ic){
                    ic.classList.toggle('dashicons-visibility', to === 'text');
                    ic.classList.toggle('dashicons-hidden',     to === 'password');
                }
            });
        }

        // ensure after DOM ready (and still works if already loaded)
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', bindHandlers);
        } else {
            bindHandlers();
        }
    })();

    // --- Domain Validation on Save ---
jQuery(document).ready(function($){
    $('form[action="options.php"]').on('submit', function(e){
        var input = $('#ss_domain');
        if(!input.length) return; // skip if field missing

        var entered = input.val().trim().toLowerCase();
        var current = window.location.hostname.toLowerCase();

        // Strip www., http(s)://, and trailing slashes
        entered = entered.replace(/^https?:\/\//, '').replace(/^www\./, '').replace(/\/+$/, '');
        current = current.replace(/^www\./, '');

        if (entered !== current) {
            e.preventDefault();
            alert("❌ Domain mismatch: Entered domain '" + entered + "' does not match your actual site domain '" + current + "'.");
            input.focus();
            input.css({'border-color':'#d63638', 'background':'#fff5f5'});
            return false;
        }
    });
});

    </script>
    <?php
}

}
