<?php
class SS_Visibility {
    const TRANSIENT_KEY = 'ss_last_visibility_result';
    // Client policy: cache last visibility result for 6 hours
    const TTL = 6 * HOUR_IN_SECONDS;

    public static function init(){
        add_action('admin_post_ss_visibility_score', [__CLASS__, 'handle_score']);
    }

    public static function handle_score(){
        if (!current_user_can('manage_options')) wp_die('No permission.');
        check_admin_referer('ss_visibility_score');

        // Always clear last result so UI never shows stale data
        delete_transient(self::TRANSIENT_KEY);

        $testUrl = isset($_POST['ss_visibility_url']) ? esc_url_raw(trim($_POST['ss_visibility_url'])) : '';
        if (!$testUrl){
            set_transient(self::TRANSIENT_KEY, ['error'=>'Please enter a URL to test.'], MINUTE_IN_SECONDS);
            wp_safe_redirect(admin_url('admin.php?page=searchshifter')); exit;
        }

        $apiKey = trim((string) get_option(SS_Settings::OPT_API_KEY, ''));
        $domain = trim((string) get_option(SS_Settings::OPT_DOMAIN, parse_url(get_site_url(), PHP_URL_HOST)));
        $base   = SS_Licensing::api_base();

        if (!$apiKey || !$domain || !$base){
            set_transient(self::TRANSIENT_KEY, ['error'=>'Missing API Base / API Key / Domain'], MINUTE_IN_SECONDS);
            wp_safe_redirect(admin_url('admin.php?page=searchshifter')); exit;
        }

        // ---- Use Http wrapper so retries/telemetry are handled centrally ----
        $resp = \SS\Core\Http::postJson('visibility.score', '/visibility/score', [
            'url'            => $testUrl,
            'domain'         => $domain,
            'plugin_version' => defined('SEARCHSHIFTER_VERSION') ? SEARCHSHIFTER_VERSION : 'unknown',
        ], ['timeout'=>20]);

        if (empty($resp['ok']) || empty($resp['json']['ok']) || empty($resp['json']['data'])) {
            $err = !empty($resp['error']) ? $resp['error'] : 'Invalid response from visibility API';
            set_transient(self::TRANSIENT_KEY, ['error'=>$err], MINUTE_IN_SECONDS);
            wp_safe_redirect(admin_url('admin.php?page=searchshifter')); exit;
        }

        $data = $resp['json']['data'];

        // UI score
        $score = isset($data['score']) ? (int)$data['score'] : 0;

        // Respect: hide author_org_schema_present
        $signals = isset($data['signals']) && is_array($data['signals']) ? $data['signals'] : [];
        unset($signals['author_org_schema_present']);

        // Recommendations (from server if provided)
        $recs = [];
        if (!empty($data['recommendations']) && is_array($data['recommendations'])) {
            $recs = $data['recommendations'];
        } else {
            $bad = static fn($v) => in_array($v, ['missing','invalid_or_missing','needs_cleanup'], true);
            if (isset($signals['qa_blocks_present'])           && $bad($signals['qa_blocks_present']))            $recs[] = 'Add Q&A blocks to this page.';
            if (isset($signals['faq_schema_valid'])            && $bad($signals['faq_schema_valid']))            $recs[] = 'Enable/repair FAQ schema on this page.';
            if (isset($signals['llms_txt_present'])            && $bad($signals['llms_txt_present']))            $recs[] = 'Create and serve /llms.txt.';
            if (isset($signals['freshness_timestamp_present']) && $bad($signals['freshness_timestamp_present'])) $recs[] = 'Add or update a freshness timestamp.';
            if (isset($signals['anchors_clean'])               && $bad($signals['anchors_clean']))               $recs[] = 'Clean up excessive in-page anchor links.';
            if (!$recs) $recs = ['Looks good. Keep content fresh and concise.'];
        }

        set_transient(self::TRANSIENT_KEY, [
            'score'    => $score,
            'ui_score' => $score,
            'signals'  => $signals,
            'url'      => (string)($data['url'] ?? $testUrl),
            'recs'     => $recs,
        ], self::TTL);

        wp_safe_redirect(admin_url('admin.php?page=searchshifter')); exit;
    }

    public static function last_result(): array {
        return get_transient(self::TRANSIENT_KEY) ?: [];
    }
}
